package gov.va.med.term.mvx.mojo;

import java.io.File;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Optional;
import java.util.UUID;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import gov.va.med.term.mvx.data.MVXCodes;
import gov.va.med.term.mvx.data.MVXCodes.MVXInfo;
import gov.va.med.term.mvx.data.MVXCodesHelper;
import gov.va.med.term.mvx.propertyTypes.PT_Annotations;
import gov.va.med.term.mvx.propertyTypes.PT_Descriptions;
import gov.va.med.term.mvx.reader.MVXReader;
import gov.va.oia.terminology.converters.sharedUtils.ComponentReference;
import gov.va.oia.terminology.converters.sharedUtils.ConsoleUtil;
import gov.va.oia.terminology.converters.sharedUtils.ConverterBaseMojo;
import gov.va.oia.terminology.converters.sharedUtils.IBDFCreationUtility;
import gov.va.oia.terminology.converters.sharedUtils.IBDFCreationUtility.DescriptionType;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Refsets;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.PropertyType;
import gov.va.oia.terminology.converters.sharedUtils.stats.ConverterUUID;
import gov.vha.isaac.MetaData;
import gov.vha.isaac.ochre.api.State;
import gov.vha.isaac.ochre.api.component.concept.ConceptChronology;
import gov.vha.isaac.ochre.api.component.concept.ConceptVersion;
import gov.vha.isaac.ochre.api.constants.DynamicSememeConstants;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeStringImpl;

/**
 * 
 * {@link MVXImportMojo}
 *
 * Goal which converts MVX data into the workbench jbin format
 * 
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 *
 */
@Mojo (name = "convert-MVX-to-ibdf", defaultPhase = LifecyclePhase.PROCESS_SOURCES)
public class MVXImportMojo extends ConverterBaseMojo
{
	private IBDFCreationUtility importUtil_;

	private HashMap<UUID, String> loadedConcepts = new HashMap<>();

	private PropertyType attributes_;
	private BPT_Refsets refsets_;
	private PT_Descriptions descriptions_;

	private UUID allMvxConceptsRefset;

	private int conceptCount = 0;

	@Override
	public void execute() throws MojoExecutionException
	{
		try
		{
			super.execute();

			//There is no global release date for mvx - but each item has its own date.  This date will only be used for metadata.
			Date date = new Date(); 
			
			importUtil_ = new IBDFCreationUtility(
					Optional.of("MVX" + " " + converterSourceArtifactVersion),
					Optional.of(MetaData.MVX_MODULES),
					outputDirectory,
					converterOutputArtifactId,
					converterOutputArtifactVersion,
					converterOutputArtifactClassifier, false, date.getTime());
			
			attributes_ = new PT_Annotations();
			descriptions_ = new PT_Descriptions();
			
			refsets_ = new BPT_Refsets("MVX");
			refsets_.addProperty("All MVX Concepts");

			// Every time concept created add membership to "All MVX Concepts"
			allMvxConceptsRefset = refsets_.getProperty("All MVX Concepts").getUUID();
			
			final MVXReader importer = new MVXReader(inputFileLocation);
			final MVXCodes terminology = importer.process();
			
			ConsoleUtil.println("Loaded Terminology containing " + terminology.getMVXInfo().size() + " entries");

			/*
			 * Methods from MVXCodes.MVXInfo:
			 * MVX_CODE() // String (MetaData.MVX_CODE)
			 * getShortDescription() // Required String FSN description?
			 * getFullVaccinename() // Required String preferred term description?
			 * getNotes() // Optional String comment
			 * getOchreState() // Required State (ACTIVE or INACTIVE) 
			 * getLastUpdatedDate(), // Required date ?
			 */
			// Parent mvxMetadata ComponentReference
			final ComponentReference mvxMetadata = ComponentReference.fromConcept(
					createType(MetaData.SOLOR_CONTENT_METADATA.getPrimordialUuid(), "MVX Metadata" + IBDFCreationUtility.metadataSemanticTag_));
			
			// loadTerminologyMetadataAttributes onto mvxMetadata
			importUtil_.loadTerminologyMetadataAttributes(converterSourceArtifactVersion, 
					Optional.empty(), converterOutputArtifactVersion, Optional.ofNullable(converterOutputArtifactClassifier), converterVersion);

			// load metadata
			importUtil_.loadMetaDataItems(Arrays.asList(attributes_, refsets_, descriptions_), mvxMetadata.getPrimordialUuid());
			
			ConsoleUtil.println("Metadata summary:");
			for (String s : importUtil_.getLoadStats().getSummary())
			{
				ConsoleUtil.println("  " + s);
			}
			importUtil_.clearLoadStats();

			// Create MVX root concept under ISAAC_ROOT
			final ConceptChronology<? extends ConceptVersion<?>> mvxRootConcept = importUtil_.createConcept("MVX", true, MetaData.ISAAC_ROOT.getPrimordialUuid());
			ConsoleUtil.println("Created MVX root concept " + mvxRootConcept.getPrimordialUuid() + " under ISAAC_ROOT");
			
			final UUID fsnSourceDescriptionTypeUUID = PT_Descriptions.Descriptions.ManufacturerName.getProperty().getUUID();
			final UUID mvxCodePropertyUuid = PT_Annotations.Attribute.MVX_CODE.getProperty().getUUID();
			
			UUID languageCode = MetaData.ENGLISH_LANGUAGE.getPrimordialUuid();
			UUID dialect = MetaData.US_ENGLISH_DIALECT.getPrimordialUuid();
			UUID caseSignificance = MetaData.DESCRIPTION_CASE_SENSITIVE.getPrimordialUuid();

			for (MVXInfo row : terminology.getMVXInfo()) {
				try {
					String code = MVXCodesHelper.getMvxCode(row) + "";
					String fsn = MVXCodesHelper.getManufacturerName(row);
					State state = MVXCodesHelper.getState(row);
					long lastUpdated = MVXCodesHelper.getLastUpdatedDate(row).getTime();

					// Create row concept
					UUID rowConceptUuid = ConverterUUID.createNamespaceUUIDFromString(code);
					final ConceptChronology<? extends ConceptVersion<?>> rowConcept = importUtil_.createConcept(rowConceptUuid, lastUpdated, state, null);
					final ComponentReference rowComponentReference = ComponentReference.fromConcept(rowConcept);
					importUtil_.addParent(rowComponentReference, mvxRootConcept.getPrimordialUuid());
					
					importUtil_.addDescription(
							rowComponentReference,
							null,
							fsn,
							DescriptionType.FSN,
							true,
							dialect,
							caseSignificance,
							languageCode,
							null,
							fsnSourceDescriptionTypeUUID,
							null,
							lastUpdated);

					// Add required MVXCode annotation
					importUtil_.addStaticStringAnnotation(rowComponentReference, code, mvxCodePropertyUuid, null);

					// Add optional Notes comment annotation
					if (StringUtils.isNotBlank(MVXCodesHelper.getNotes(row))) {
						importUtil_.addAnnotation(rowComponentReference, null, new DynamicSememeStringImpl(MVXCodesHelper.getNotes(row)), 
								DynamicSememeConstants.get().DYNAMIC_SEMEME_COMMENT_ATTRIBUTE.getPrimordialUuid(), null, lastUpdated);
					}

					// Add to refset allMvxConceptsRefset
					importUtil_.addRefsetMembership(rowComponentReference, allMvxConceptsRefset, null, lastUpdated);

					++conceptCount;
				} catch (Exception e) {
					final String msg = "Failed processing row with " + e.getClass().getSimpleName() + " " + e.getLocalizedMessage() + ": " + row;
					ConsoleUtil.printErrorln(msg);
					throw new RuntimeException(msg, e);
				}
			}
			ConsoleUtil.println("Processed " + conceptCount + " concepts");
			
			ConsoleUtil.println("Load Statistics");

			for (String line : importUtil_.getLoadStats().getSummary())
			{
				ConsoleUtil.println(line);
			}
			// this could be removed from final release. Just added to help debug editor problems.
			ConsoleUtil.println("Dumping UUID Debug File");
			ConverterUUID.dump(outputDirectory, "mvxUuid");

			importUtil_.shutdown();
			ConsoleUtil.writeOutputToFile(new File(outputDirectory, "ConsoleOutput.txt").toPath());
		}
		catch (Exception ex)
		{
			throw new MojoExecutionException(ex.getLocalizedMessage(), ex);
		}
	}

	private ConceptChronology<? extends ConceptVersion<?>> createType(UUID parentUuid, String typeName) throws Exception
	{
		ConceptChronology<? extends ConceptVersion<?>> concept = importUtil_.createConcept(typeName, true);
		loadedConcepts.put(concept.getPrimordialUuid(), typeName);
		importUtil_.addParent(ComponentReference.fromConcept(concept), parentUuid);
		return concept;
	}

	public static void main(String[] args) throws MojoExecutionException
	{
		MVXImportMojo i = new MVXImportMojo();
		i.outputDirectory = new File("../mvx-ibdf/target");
		i.inputFileLocation = new File("../mvx-ibdf/target/generated-resources/src/");
		i.converterOutputArtifactVersion = "2016.01.07.foo";
		i.converterVersion = "SNAPSHOT";
		i.converterSourceArtifactVersion = "17.0";
		i.execute();
	}
}